<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:fmp="http://www.filemaker.com/fmpxmlresult" exclude-result-prefixes="fmp">
	<xsl:output method="html" version="1.0" encoding="UTF-8" indent="yes"/>
	<!--
File: complex_table.xsl

Transforms data exported in the FMPXMLRESULT grammar into an 
HTML table. Variables are provided to customize the color of 
headings and odd/even rows.

===============================================================

Copyright © 2002 FileMaker, Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in 
  the documentation and/or other materials provided with the
  distribution.

* Neither the name of the FileMaker, Inc. nor the names of its 
  contributors may be used to endorse or promote products derived
  from this software without specific prior written
  permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
    
===============================================================
-->
	<!--
A slightly more involved stylesheet that builds an HTML table using multiple 
templates and more sophisticated select statements.
-->
	<xsl:template match="fmp:FMPXMLRESULT">
		<html>
			<head>
				<title>
					<xsl:value-of select="$dbName"/>
				</title>
			</head>
			<body>
				<table border="0" cellPadding="3" cellSpacing="1">
					<xsl:call-template name="header"/>
					<!--
This next block simply alternates the colors like this:
					
MainOne | ColorOne
MainTwo | ColorTwo
MainOne | ColorOne
MainTwo | ColorTwo
                                  
using the position() mod 2, checking for odd or
even rows.
-->
					<xsl:for-each select="fmp:RESULTSET/fmp:ROW">
						<xsl:variable name="id-background-color">
							<xsl:choose>
								<xsl:when test="position() mod 2 = 1">
									<xsl:value-of select="$mainOne"/>
								</xsl:when>
								<xsl:when test="position() mod 2 = 0">
									<xsl:value-of select="$mainTwo"/>
								</xsl:when>
							</xsl:choose>
						</xsl:variable>
						<xsl:variable name="data-background-color">
							<xsl:choose>
								<xsl:when test="position() mod 2 = 1">
									<xsl:value-of select="$colorOne"/>
								</xsl:when>
								<xsl:when test="position() mod 2 = 0">
									<xsl:value-of select="$colorTwo"/>
								</xsl:when>
							</xsl:choose>
						</xsl:variable>
						<tr>
							<xsl:for-each select="fmp:COL">
								<xsl:variable name="background-color">
									<xsl:choose>
										<xsl:when test="position() = 1">
											<xsl:value-of select="$id-background-color"/>
										</xsl:when>
										<xsl:otherwise>
											<xsl:value-of select="$data-background-color"/>
										</xsl:otherwise>
									</xsl:choose>
								</xsl:variable>
								<td bgcolor="{$background-color}">
									<xsl:value-of select="fmp:DATA"/>
								</td>
							</xsl:for-each>
						</tr>
					</xsl:for-each>
					<xsl:call-template name="footer"/>
				</table>
			</body>
		</html>
	</xsl:template>
	<!--
Template: header
	        
Creates the heading for the HTML table.
	        
The database name and number of records are displayed above the fields names.
Field names are displayed in "headerFontColor" color with a cell background color
of "headerColor".
	-->
	<xsl:template name="header">
		<tr>
			<td align="middle">
				<xsl:attribute name="bgColor"><xsl:value-of select="$headerColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:call-template name="numfields"/></xsl:attribute>
				<font size="+1">
					<xsl:attribute name="color"><xsl:value-of select="$headerFontColor"/></xsl:attribute>
					<xsl:text>Database: </xsl:text>
					<xsl:value-of select="fmp:DATABASE/@NAME"/>
				</font>
			</td>
		</tr>
		<tr>
			<td align="middle">
				<xsl:attribute name="bgColor"><xsl:value-of select="$headerColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:call-template name="numfields"/></xsl:attribute>
				<font color="#ffffff" size="+1">
					<xsl:text>Records: </xsl:text>
					<xsl:value-of select="fmp:DATABASE/@RECORDS"/>
				</font>
			</td>
		</tr>
		<tr>
			<xsl:for-each select="fmp:METADATA/fmp:FIELD">
				<td align="middle">
					<xsl:attribute name="bgColor"><xsl:value-of select="$headerColor"/></xsl:attribute>
					<font size="+1">
						<xsl:attribute name="color"><xsl:value-of select="$headerFontColor"/></xsl:attribute>
						<xsl:value-of select="@NAME"/>
					</font>
				</td>
			</xsl:for-each>
		</tr>
	</xsl:template>
	<!--
Template: footer
	        
Creates the footer for the HTML table with a cell background color of "footerColor".
	        
The version of FileMaker Pro used to create the XML file is displayed along with the 
build date.
	-->
	<xsl:template name="footer">
		<tr>
			<td align="middle">
				<xsl:attribute name="bgColor"><xsl:value-of select="$footerColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:call-template name="numfields"/></xsl:attribute>
				<font color="#ffffff" size="+1">
					<xsl:text>FileMaker Pro Version: </xsl:text>
					<xsl:value-of select="fmp:PRODUCT/@VERSION"/>
				</font>
			</td>
		</tr>
		<tr>
			<td align="middle">
				<xsl:attribute name="bgColor"><xsl:value-of select="$footerColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:call-template name="numfields"/></xsl:attribute>
				<font size="+1">
					<xsl:attribute name="color"><xsl:value-of select="$headerFontColor"/></xsl:attribute>
					<xsl:text>Build: </xsl:text>
					<xsl:value-of select="fmp:PRODUCT/@BUILD"/>
				</font>
			</td>
		</tr>
	</xsl:template>
	<!--
Template: numfields

Simple utility template that checks for the number of children in the METADATA
element and returns the number of fields in the database - handy when building
tables.
-->
	<xsl:template name="numfields" match="fmp:METADATA">
		<xsl:value-of select="count(fmp:METADATA/child::*)"/>
	</xsl:template>
	<xsl:variable name="dbName">
		<xsl:value-of select="fmp:FMPXMLRESULT/fmp:DATABASE/@NAME"/>
	</xsl:variable>
	<!--
The table is arranged like this:
         
HeaderColor
MainOne | ColorOne
MainTwo | ColorTwo
MainOne | ColorOne
MainTwo | ColorTwo
FooterColor
      
Changing the values of the variables below will allow for easy modifications to the
appearance of the resulting HTML table without the need for editing the XSL
templates. 
	-->
	<xsl:variable name="mainOne">
		<xsl:text>#9bb0d4</xsl:text>
	</xsl:variable>
	<xsl:variable name="mainTwo">
		<xsl:text>#7f95d4</xsl:text>
	</xsl:variable>
	<xsl:variable name="colorOne">
		<xsl:text>#bcd0f0</xsl:text>
	</xsl:variable>
	<xsl:variable name="colorTwo">
		<xsl:text>#9cb0f0</xsl:text>
	</xsl:variable>
	<xsl:variable name="headerColor">
		<xsl:text>#000000</xsl:text>
	</xsl:variable>
	<xsl:variable name="footerColor">
		<xsl:text>#000000</xsl:text>
	</xsl:variable>
	<xsl:variable name="headerFontColor">
		<xsl:text>#ffffff</xsl:text>
	</xsl:variable>
</xsl:stylesheet>
